<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use App\Models\Examination;
use App\Models\WhatsAppTemplate;
use Illuminate\Console\Command;
use App\Services\WhatsAppService;

class SendExaminationReminders extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'reminders:send';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Send reminder messages to clients based on templates';

    protected WhatsAppService $whatsAppService;

    /**
     * Create a new command instance.
     *
     * @param WhatsAppService $whatsAppService
     */
    public function __construct(WhatsAppService $whatsAppService)
    {
        parent::__construct();
        $this->whatsAppService = $whatsAppService;
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        // Fetch active templates with age range and months not empty
        $templates = WhatsappTemplate::where('active', true)
            ->whereNotNull('age_range')
            ->whereNotNull('months')
            ->get();

        foreach ($templates as $template) {
            // If the attributes are stored as JSON strings, decode them.
            $ageRange = is_string($template->age_range) ? json_decode($template->age_range, true) : $template->age_range;
            $months = is_string($template->months) ? json_decode($template->months, true) : $template->months;

            $minAge = $ageRange[0] ?? null;
            $maxAge = $ageRange[1] ?? null;

            foreach ($months as $month) {
                $targetDate = Carbon::now()->subMonths($month);
                // Carbon::today()
                // $targetDate->toDateString()
                $examinations = Examination::whereDate('created_at', '=',$targetDate->toDateString())
                    ->whereHas('client', function ($query) use ($minAge, $maxAge) {
                        $query->where(function ($ageQuery) use ($minAge, $maxAge) {
                            if ($minAge) {
                                $minBirthDate = Carbon::now()->subYears($minAge)->endOfDay();
                                $ageQuery->where('birth_date', '<=', $minBirthDate);
                            }
                            if ($maxAge) {
                                $maxBirthDate = Carbon::now()->subYears($maxAge)->startOfDay();
                                $ageQuery->where('birth_date', '>=', $maxBirthDate);
                            }
                        });
                    })
                    ->whereIn('id', function ($subquery) {
                        $subquery->selectRaw('MAX(id)')
                            ->from('examinations')
                            ->groupBy('client_id');
                    })
                    ->get();


                foreach ($examinations as $examination) {
                    $client = $examination->client;

                    $placeholders = [
                        'name' => $client->first_name,
                        'date' => $examination->created_at->format('Y-m-d'),
                    ];

                    try {
                        $this->whatsAppService->sendTemplateMessage(
                            $template->code,
                            $client->phone,
                            $placeholders
                        );

                        $this->info("Reminder sent to {$client->first_name} ({$client->phone}) for examination {$examination->id}");
                    } catch (\Exception $e) {
                        $this->error("Failed to send message to {$client->phone}: {$e->getMessage()}");
                    }
                }
            }
        }


        return Command::SUCCESS;
    }
}
