<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use App\Models\City;
use App\Models\Client;
use App\Models\Region;
use App\Models\Country;
use App\Models\Examination;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;

class ClientsController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:clients-list', ['only' => ['index']]);
        $this->middleware('permission:clients-show', ['only' => ['show']]);
        $this->middleware('permission:clients-create', ['only' => ['create', 'store']]); // يجب التأكد من وجود هذه السطر
        $this->middleware('permission:clients-edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:clients-delete', ['only' => ['destroy']]);
    }


    public function index()
    {
        $clients = Client::all();

        return view('admin.clients.index', compact('clients'));
    }

    public function create()
    {
        $cities = City::with('regions')->get();
        $countries = Country::all();
        return view('admin.clients.create', compact('cities', 'countries'));
    }



    public function store(Request $request)
    {


        $request->validate([
            'first_name'   => 'required|string|max:255',
            'last_name'    => 'required|string|max:255',
            'age'   => 'required',
            'address'      => 'nullable|string|max:255',
            'phone'        => 'required|string|max:20',
            'city_id'      => 'nullable|exists:cities,id',
            'region_id'    => 'nullable|exists:regions,id',
            'gender'       => 'nullable',
            'country_code' => 'required|string',
            // Custom validation for unique combination of first_name and last_name
            'first_name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('clients')->where(function ($query) use ($request) {
                    return $query->where('last_name', $request->last_name);
                }),
            ],
        ], [
            'first_name.unique' => __('The combination of first name and last name must be unique.'),
        ]);




        // Process phone number and other logic as needed
        $countryCode = preg_replace('/\D/', '', $request->country_code);
        $phoneNumber = ltrim($request->phone, '0');
        $fullPhoneNumber = '+' . $countryCode . $phoneNumber;

        // Calculate the age


        // Save the client data
        Client::create([
            'first_name'        => $request->first_name,
            'last_name'         => $request->last_name,
            'age'               =>$request->age,
            'address'           => $request->address,
            'phone'             => $fullPhoneNumber,
            'city_id'           => $request->city_id,
            'gender'            => $request->gender,
            'region_id'         => $request->region_id,
            'country_code'      => $request->country_code,
            'full_phone_number' => $fullPhoneNumber,
        ]);

        session()->flash('success', __('messages.create'));
        return redirect()->route('admin.clients.index');
    }




    public function show($id)
    {
        $client = Client::findOrFail($id);
        $examinations = Examination::where('client_id', $id)->get();

        return view('admin.clients.show', compact('examinations', 'client'));
    }






    public function edit($id)
    {
        // Fetch client, cities, and related regions
        $client = Client::findOrFail($id);
        $cities = City::with('regions')->get();
        $countries = Country::all();
        // return $client ;

        return view('admin.clients.edit', compact('client', 'cities', 'countries'));
    }


    public function update(Request $request, $id)
    {
        $request->validate([
            'first_name'   => 'required|string|max:255',
            'last_name'    => 'required|string|max:255',
            'age'   => 'nullable',
            'address'      => 'nullable|string|max:255',
            'phone'        => 'required|string|max:20',
            'city_id'      => 'nullable|exists:cities,id',
            'region_id'    => 'nullable|exists:regions,id',
            'gender'       => 'nullable',
            'country_code' => 'required|string',

        ]);

        // جلب العميل من قاعدة البيانات
        $client = Client::findOrFail($id);

        // التحقق إذا كان الاسم الأول والاسم الأخير معًا مسجلين مسبقًا في النظام
        $existingClient = Client::where('first_name', $request->first_name)
            ->where('last_name', $request->last_name)
            ->where('id', '!=', $client->id) // استبعاد العميل الحالي من التحقق
            ->first();

        if ($existingClient) {
            return back()->withErrors([
                'first_name' =>  __('translations.name_exists'),
            ]);
        }

        // التحقق إذا كان أحدهما فقط مسجلًا مسبقًا (الاسم الأول أو الاسم الأخير)
        $existingFirstName = Client::where('first_name', $request->first_name)
            ->where('id', '!=', $client->id)
            ->first();

        $existingLastName = Client::where('last_name', $request->last_name)
            ->where('id', '!=', $client->id)
            ->first();

        // الآن، دمج country_code و phone بشكل صحيح
        // إزالة أي أحرف غير رقمية من country_code
        $countryCode = preg_replace('/\D/', '', $request->country_code); // إزالة أي أحرف غير رقمية

        // تنسيق رقم الهاتف (إزالة الصفر الأول إذا كان موجودًا)
        $phoneNumber = ltrim($request->phone, '0'); // إزالة الصفر الأول من رقم الهاتف

        // دمج الرقم مع رمز الدولة
        $fullPhoneNumber =  $phoneNumber;



        // تحديث العميل بناءً على التحقق من الاسم
        if ($existingFirstName && !$existingLastName) {
            // إذا كان الاسم الأول موجودًا فقط، التحديث مسموح
            $client->update([
                'first_name'  => $request->first_name,
                'last_name'   => $request->last_name,
                'phone'       => $fullPhoneNumber, // تحديث رقم الهاتف الكامل
                'full_phone_number' => $fullPhoneNumber, // إضافة الرقم الكامل
                'age'         => $request->age,
                'address'     => $request->address,
                'city_id'     => $request->city_id,
                'region_id'   => $request->region_id,
                'gender'      => $request->gender,
            ]);
        } elseif ($existingLastName && !$existingFirstName) {
            // إذا كان الاسم الأخير موجودًا فقط، التحديث مسموح
            $client->update([
                'first_name'  => $request->first_name,
                'last_name'   => $request->last_name,
                'phone'       => $fullPhoneNumber, // تحديث رقم الهاتف الكامل
                'full_phone_number' => $fullPhoneNumber, // إضافة الرقم الكامل
                'age'         => $request->age,
                'address'     => $request->address,
                'city_id'     => $request->city_id,
                'region_id'   => $request->region_id,
                'gender'      => $request->gender,
            ]);
        } else {
            // في حال عدم وجود أية مشكلة، يتم تحديث العميل
            $client->update([
                'first_name'  => $request->first_name,
                'last_name'   => $request->last_name,
                'phone'       => $fullPhoneNumber, // تحديث رقم الهاتف الكامل
                'full_phone_number' => $fullPhoneNumber, // إضافة الرقم الكامل

                'age'         => $request->age,
                'address'     => $request->address,
                'city_id'     => $request->city_id,
                'region_id'   => $request->region_id,
                'gender'      => $request->gender,
            ]);
        }

        // إعادة التوجيه مع رسالة نجاح
        return redirect()->route('admin.clients.index')->with('success', 'تم تحديث العميل بنجاح');
    }







    public function destroy(int $id)
    {
        $client = Client::findOrFail($id);
        $client->delete();

        session()->flash('success', __('messages.delete'));
        return redirect()->route('admin.clients.index');
    }



    public function getRegionsByCity($cityId)
    {
        $regions = Region::where('city_id', $cityId)->get();

        return response()->json($regions);
    }
}
