<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class DelayedCancelledController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:delayed-list', ['only' => ['index']]);
        $this->middleware('permission:delayed-create', ['only' => ['create', 'store']]);
        $this->middleware('permission:delayed-edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:delayed-delete', ['only' => ['destroy']]);
    }

    // Display a listing of the delayed with examination status "ارسال الي المشغل"
    public function index()
    {
        // تحديد التاريخ الحالي وطرح 30 يوم
        $thirtyDaysAgo = Carbon::now()->subDays(30);

        // استرجاع الفحوصات التي تجاوزت 30 يومًا
        $examinations = Examination::with('client')
            ->where('status', 'في انتظار التاكيد')
            ->where('created_at', '<', $thirtyDaysAgo) // الفحص الذي تم إنشاؤه قبل أكثر من 30 يومًا
            ->latest()
            ->get();

        return view('admin.delayed.index', compact('examinations'));
    }



    // Show the form for creating a new examination
    public function create()
    {
        $clients = Client::all(); // Fetch all clients
        $types = ['نظارات', 'عدسات']; // Example types
        $examination_count = Examination::count();
        $examination_code = 'E-' . str_pad($examination_count + 1, 4, '0', STR_PAD_LEFT);
        $examiners = Examiner::all();
        return view('admin.delayed.create', compact('clients', 'types', 'examination_code', 'examiners'));
    }

    public function store(Request $request)
    {
        // Validate the request data
        $request->validate([
            'code' => 'required|unique:examinations,code',
            'notes' => 'nullable|string|max:255',
            'type' => 'required|string',  // This is the type for the examination (glasses or lenses)
            'client_id' => 'required|exists:clients,id',
            'examiner_id' => 'required|exists:examiners,id',
            'status' => 'nullable|string|max:255', // Add status validation if needed

            // Validation for right and left eye fields
            'rt_sph' => 'required|string',
            'rt_cyl' => 'required|string',
            'rt_axis' => 'required|string',
            'rt_lpd' => 'required|string',
            'rt_add' => 'required|string',
            'lt_sph' => 'required|string',
            'lt_cyl' => 'required|string',
            'lt_axis' => 'required|string',
            'lt_lpd' => 'required|string',
            'lt_add' => 'required|string',
        ]);

        // Prepare the request data for the examination
        $requestData = $request->only(['code', 'notes', 'type', 'client_id', 'examiner_id']);
        $requestData['status'] = 'في انتظار التاكيد';  // Default status
        $requestData['create_by'] = auth()->user()->id;
        $requestData['update_by'] = auth()->user()->id;
        $requestData['make_from'] = 'delayed';

        // Create the examination record
        $examination = Examination::create($requestData);

        // Create details for the right eye (RT)
        $rtDetailData = [
            'sph' => $request->input('rt_sph'),
            'cyl' => $request->input('rt_cyl'),
            'axis' => $request->input('rt_axis'),
            'lpd' => $request->input('rt_lpd'),
            'add' => $request->input('rt_add'),
            'type' => 'R',  // Right eye
            'examination_id' => $examination->id,  // Link to the examination
        ];
        Examination_detail::create($rtDetailData);

        // Create details for the left eye (LT)
        $ltDetailData = [
            'sph' => $request->input('lt_sph'),
            'cyl' => $request->input('lt_cyl'),
            'axis' => $request->input('lt_axis'),
            'lpd' => $request->input('lt_lpd'),
            'add' => $request->input('lt_add'),
            'type' => 'L',  // Left eye
            'examination_id' => $examination->id,  // Link to the examination
        ];
        Examination_detail::create($ltDetailData);

        // Flash success message and redirect
        session()->flash('success', __('messages.create'));
        return redirect()->route('admin.delayed.index');
    }

    public function edit($id)
    {
        // الحصول على السجل الذي نريد تعديله مع تفاصيله
        $examination = Examination::with('details')->findOrFail($id);

        // استرجاع البيانات الأخرى (العملاء، الفاحصين، الأنواع)
        $clients = Client::all();
        $examiners = Examiner::all();
        $types = ['نظارات', 'عدسات']; // أو يمكنك استرجاع الأنواع من قاعدة البيانات إذا كانت متغيرة

        // تمرير البيانات إلى صفحة التعديل
        return view('admin.delayed.edit', compact('examination', 'clients', 'examiners', 'types'));
    }



    public function show($id)
    {
        // الحصول على السجل الذي نريد تعديله مع تفاصيله
        $examination = Examination::with('details')->findOrFail($id);

        // استرجاع البيانات الأخرى (العملاء، الفاحصين، الأنواع)
        $clients = Client::all();
        $examiners = Examiner::all();
        $types = ['نظارات', 'عدسات']; // أو يمكنك استرجاع الأنواع من قاعدة البيانات إذا كانت متغيرة

        // تمرير البيانات إلى صفحة العرض
        return view('admin.delayed.show', compact('examination', 'clients', 'examiners', 'types'));
    }


    public function show_examination(int $id)
    {
        // Retrieve the examination record by its ID
        $examination = Examination::where('id', $id)->with('details')->first();
        $client = $examination->client;

        // Fetch the data based on room_code for the specific examination
        $user = Auth::user();
        $data = Data::where('room_code', $user->room_code)->get();
        $rooms = Room::all();

        // Prepare an array to hold formatted data
        $formatted_data = [];

        // Format the data as needed (similar to the create method)
        foreach ($data as $item) {
            // Parse the right eye (data_rt)
            preg_match('/R\s*-\s*-\s*-\s*(\d+\.\d+)\s*-\s*-\s*-\s*(\d+\.\d+)\s*-\s*(\d+)/', $item->data_rt, $matches_rt);
            $formatted_rt = [
                'type' => 'R',
                'sph' => $matches_rt[1] ?? 'N/A',  // SPH: -4.50
                'cyl' => $matches_rt[2] ?? 'N/A',  // CYL: -1.00
                'axis' => $matches_rt[3] ?? 'N/A', // Axis: 175
                'lpd' => $item->lpd_rt ?? 'N/A',   // Lens Power Difference for right eye
                'add' => $item->add_rt ?? 'N/A',   // Addition for right eye
            ];

            // Parse the left eye (data_lt)
            preg_match('/L\s*-\s*-\s*-\s*(\d+\.\d+)\s*-\s*-\s*-\s*(\d+\.\d+)\s*-\s*(\d+)/', $item->data_lt, $matches_lt);
            $formatted_lt = [
                'type' => 'L',
                'sph' => $matches_lt[1] ?? 'N/A',  // SPH: -4.25
                'cyl' => $matches_lt[2] ?? 'N/A',  // CYL: -1.75
                'axis' => $matches_lt[3] ?? 'N/A', // Axis: 170
                'lpd' => $item->lpd_lt ?? 'N/A',   // Lens Power Difference for left eye
                'add' => $item->add_lt ?? 'N/A',   // Addition for left eye
            ];

            // Collect formatted data for each item
            $formatted_data[] = [
                'id' => $item->id,
                'room_code' => $item->room_code,
                'right_eye' => $formatted_rt,
                'left_eye' => $formatted_lt,
            ];
        }

        // Fetch other required data for the view
        $clients = Client::all(); // Fetch all clients
        $types = ['نظارات', 'عدسات']; // Example types
        $examiners = Examiner::all();

        // Return the formatted data along with the rest of the information to the view
        return view('admin.delayed.show', compact('examination', 'client', 'clients', 'types', 'examiners', 'formatted_data', 'rooms'));
    }

    public function update(Request $request, $id)
    {
        // Validate the request data
        $request->validate([
            'code' => 'required|unique:examinations,code,' . $id,
            'notes' => 'nullable|string|max:255',
            'type' => 'required|string',
            'client_id' => 'required|exists:clients,id',
            'examiner_id' => 'required|exists:examiners,id',
            'rt_sph' => 'required|string',
            'rt_cyl' => 'required|string',
            'rt_axis' => 'required|string',
            'rt_lpd' => 'required|string',
            'rt_add' => 'required|string',
            'lt_sph' => 'required|string',
            'lt_cyl' => 'required|string',
            'lt_axis' => 'required|string',
            'lt_lpd' => 'required|string',
            'lt_add' => 'required|string',
        ]);

        // Update the examination table
        $examination = Examination::findOrFail($id);
        $examination->update($request->only(['code', 'notes', 'type', 'client_id', 'examiner_id']));

        // Handle the RT and LT details update or creation
        // Update or create RT details
        $examination->details()->updateOrCreate(
            ['type' => 'R', 'examination_id' => $id],
            [
                'sph' => $request->input('rt_sph'),
                'cyl' => $request->input('rt_cyl'),
                'axis' => $request->input('rt_axis'),
                'lpd' => $request->input('rt_lpd'),
                'add' => $request->input('rt_add')
            ]
        );

        // Update or create LT details
        $examination->details()->updateOrCreate(
            ['type' => 'L', 'examination_id' => $id],
            [
                'sph' => $request->input('lt_sph'),
                'cyl' => $request->input('lt_cyl'),
                'axis' => $request->input('lt_axis'),
                'lpd' => $request->input('lt_lpd'),
                'add' => $request->input('lt_add')
            ]
        );

        session()->flash('success', __('messages.update'));
        return redirect()->route('admin.delayed.index');
    }







    // public function updateStatus(Request $request, $id)
    // {
    //     // العثور على الفحص بواسطة الـ ID مع تضمين العميل
    //     $examination = Examination::with('client')->findOrFail($id);

    //     // تحديث حالة الفحص
    //     $examination->status = $request->input('status');
    //     $examination->save();

    //     // التحقق من نجاح عملية الحفظ
    //     if ($examination->wasChanged()) {
    //         // إرسال الـ Job إلى الـ Queue مباشرة بعد تحديث الحالة
    //         SendWhatsappMessage::dispatch($examination);

    //         // العودة إلى صفحة الاستقبال مع رسالة نجاح
    //         return redirect()->route('admin.delayed.index')->with('success', 'تم تحديث الحالة بنجاح. سيتم إرسال رسالة WhatsApp قريباً.');
    //     }

    //     // العودة إلى صفحة الاستقبال مع رسالة نجاح إذا لم يكن هناك تغيير
    //     return redirect()->route('admin.delayed.index')->with('success', 'تم تحديث الحالة بنجاح.');
    // }

    public function updateStatus(Request $request, $id)
    {
        // العثور على الفحص باستخدام الـ ID
        $examination = Examination::findOrFail($id);

        // تحديث الحالة فقط إذا تم إرسالها في الـ request
        if ($request->has('status')) {
            $examination->status = $request->input('status');
        }

        // إضافة أي تحديثات أخرى على الـ Examination إذا كان هناك المزيد من الحقول.
        // مثال:
        // if ($request->has('other_field')) {
        //    $examination->other_field = $request->input('other_field');
        // }

        // حفظ التغييرات
        $examination->save();

        // تحقق إذا كانت الحالة قد تغيرت فعلاً
        if ($examination->wasChanged()) {
            // إرسال رسالة WhatsApp عبر الـ Queue فقط إذا كانت الحالة قد تغيرت
            SendWhatsappMessage::dispatch($examination);

            return redirect()->route('admin.delayed.index')->with('success', 'تم تحديث الحالة وسيتم إرسال رسالة WhatsApp.');
        }

        // إذا لم تتغير الحالة
        return redirect()->route('admin.delayed.index')->with('info', 'الحالة لم تتغير.');
    }





    public function print($id)
    {
        $examination = Examination::with(['client', 'examiner'])->findOrFail($id);
        return view('admin.delayed.print', compact('examination'));
    }

    public function printSpecial($id)
    {
        $examination = Examination::with(['client', 'examiner'])->findOrFail($id);

        // تأكد من أن الفحص مميز
        if ($examination->special) {
            return view('admin.delayed.print_special', compact('examination'));
        } else {
            return view('admin.delayed.print', compact('examination'));
        }
    }
}
