<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;

class ProductController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:products-list', ['only' => ['index']]);
        $this->middleware('permission:products-create', ['only' => ['create', 'store']]);
        $this->middleware('permission:products-edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:products-delete', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $products = Product::all();
        return view('admin.products.index', compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Category::all();
        return view('admin.products.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'price' => 'required|numeric',
            'description' => 'nullable|string',
            'image' => 'nullable',
            'category_id' => 'required|exists:categories,id',
        ]);

        $product = Product::create($request->all());
        if ($request->hasFile('image')) {
            $pictureName = time() . '.' . $request->file('image')->getClientOriginalExtension();
            $picturePath = $request->file('image')->storeAs('products', $pictureName, 'custom_path');
            $requestData['image'] = $picturePath;
            $product->image = $picturePath;
        }
        $product->save();
        return redirect()->route('admin.products.index')->with('success', __('translations.product-created-successfully'));
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $product = Product::findOrFail($id);
        return view('admin.products.show', compact('product'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $product = Product::findOrFail($id);
        $categories = Category::all();
        return view('admin.products.edit', compact('product', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $product = Product::findOrFail($id);
        $product->update($request->all());
        if ($request->hasFile('image')) {
            $pictureName = time() . '.' . $request->file('image')->getClientOriginalExtension();
            $picturePath = $request->file('image')->storeAs('products', $pictureName, 'custom_path');
            $requestData['image'] = $picturePath;
            $product->image = $picturePath;
        }
        $product->save();
        return redirect()->route('admin.products.index')->with('success', __('translations.product-updated-successfully'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $product = Product::findOrFail($id);
        $product->delete();
        return redirect()->route('admin.products.index')->with('success', __('translations.product-deleted-successfully'));
    }
}
