<?php

namespace App\Http\Controllers\Admin;

use App\Models\Data;
use App\Models\Room;
use App\Models\Client;
use App\Models\Examiner;
use App\Models\Whatsapp;
use App\Models\Reception;
use App\Models\Examination;
use Illuminate\Http\Request;
use App\Models\WhatsAppTemplate;
use App\Jobs\SendWhatsappMessage;
use App\Services\WhatsAppService;
use App\Models\Examination_detail;
use App\Http\Controllers\Controller;
use App\Models\Settings;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

use Carbon\Carbon;

class ReceptionController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:receptions-list', ['only' => ['index']]);
        $this->middleware('permission:receptions-create', ['only' => ['create', 'store']]);
        $this->middleware('permission:receptions-edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:receptions-delete', ['only' => ['destroy']]);
    }



    // Display a listing of the examinations
    public function index()
    {
        // تحديد التاريخ الحالي وطرح 30 يوم
        $thirtyDaysAgo = Carbon::now()->subDays(30);

        // استرجاع الفحوصات التي لم تتجاوز 30 يومًا
        $examinations = Examination::with('client')
            ->where('status', 'في انتظار التاكيد')
            ->where('created_at', '>=', $thirtyDaysAgo) // الفحص الذي تم إنشاؤه في آخر 30 يومًا
            ->latest()
            ->get();

        return view('admin.receptions.index', compact('examinations'));
    }



    // Show the form for creating a new examination
    public function create()
    {
          $user = Auth::user();
        $group=room::where('room_code',$user->room_code)->pluck('group')->first();
 $clients = Client::orderBy('created_at', 'desc')->get();

        $types = ['نظارات', 'عدسات'];
        $specials = ['عادي', 'مميز', 'مستعجل', 'تحضير', 'توصيل'];
        $examination_count = Examination::count();
$lastExamination = Examination::
    // where('make_from', 'receptions')
      where('group', 'P')
    ->latest('id')
    ->first();
  if ($lastExamination) {
    $lastCodeNumber = (int) substr($lastExamination->code, 2); // Extract numeric part after 'E-'
    if ($lastCodeNumber >= 300) {
        $examination_code = 'P'.'-'. str_pad(1, 3, '0', STR_PAD_LEFT); // Reset to 1 if greater than or equal to 300
    } else {
        $examination_code = 'P'.'-'. str_pad($lastCodeNumber + 1, 3, '0', STR_PAD_LEFT);
    }
} else {
    $examination_code = 'P'.'-'. str_pad(1, 3, '0', STR_PAD_LEFT); // Default code if no records exist
}

        $examiners = Examiner::all();
        return view('admin.receptions.create', compact('clients', 'types', 'examination_code', 'examiners','specials'));
    }

public function store(Request $request)
{
    // Validate the request data
    $request->validate([
        'code' => 'required|string',
        'notes' => 'nullable|string|max:255',
        'type' => 'required|string',
        'special' => 'required|string',
        'client_id' => 'required|exists:clients,id',
        'file' => 'nullable|file|mimes:jpg,png,pdf|max:2048', // Ensure correct file types and size
    ]);

    $lastExamination = Examination::with('details')
    ->where('client_id', $request->client_id)
    ->orderBy('created_at', 'desc')
    ->first();

    // return $lastExamination_data ;

    DB::beginTransaction(); // Start a transaction for database integrity

    try {
        // Prepare data for examination creation
        $requestData = $request->only(['notes', 'type', 'client_id', 'special', 'code']);
        $requestData['status'] = 'في انتظار التاكيد'; // "Pending Confirmation"
        $requestData['create_by'] = auth()->user()->id;
        $requestData['examiner_id'] = 12; // Default examiner
        $requestData['group'] = 'P'; // Example group
        $requestData['update_by'] = auth()->user()->id;
        $requestData['make_from'] = 'receptions';

        // Handle file upload
        if ($request->hasFile('file')) {
            $pictureName = time() . '.' . $request->file('file')->getClientOriginalExtension();
            $picturePath = $request->file('file')->storeAs('examination', $pictureName, 'custom_path');
            $requestData['file'] = $picturePath;
        }

        // return $requestData;
        // Create the examination record
        $examination = Examination::create($requestData);
// dd($examination);
        // Retrieve the last examination details if they exist






        if ($request->button_type == "save") {
            // Create new details for the right and left eye
            Examination_detail::create([
                'sph' => $request->input('rt_sph'),
                'cyl' => $request->input('rt_cyl'),
                'axis' => $request->input('rt_axis'),
                'lpd' => $request->input('rt_lpd'),
                'add' => $request->input('rt_add'),
                'type' => 'R', // Right eye
                'examination_id' => $examination->id,
            ]);

            Examination_detail::create([
                'sph' => $request->input('lt_sph'),
                'cyl' => $request->input('lt_cyl'),
                'axis' => $request->input('lt_axis'),
                'lpd' => $request->input('lt_lpd'),
                'add' => $request->input('lt_add'),
                'type' => 'L', // Left eye
                'examination_id' => $examination->id,
            ]);
        } else {
            // Use previous examination data if available
            if ($lastExamination && $lastExamination->details) {
                $rtDetail = $lastExamination->details->where('type', 'R')->first();
                $ltDetail = $lastExamination->details->where('type', 'L')->first();

                Examination_detail::create([
                    'sph' => $rtDetail ? $rtDetail->sph : $request->input('rt_sph'),
                    'cyl' => $rtDetail->cyl ,
                    'axis' => $rtDetail->axis ,
                    'lpd' => $rtDetail->lpd ,
                    'add' => $rtDetail->add ,
                    'type' => 'R', // Right eye
                    'examination_id' => $examination->id,
                ]);

                Examination_detail::create([
                    'sph' => $ltDetail ? $ltDetail->sph : $request->input('lt_sph'),
                    'cyl' => $ltDetail->cyl,
                    'axis' => $ltDetail->axis,
                    'lpd' => $ltDetail->lpd,
                    'add' =>$ltDetail->add,
                    'type' => 'L', // Left eye
                    'examination_id' => $examination->id,
                ]);
            }
        }

        DB::commit(); // Commit transaction if everything is successful

        // Flash success message and redirect
        session()->flash('success', __('messages.create'));
        return redirect()->route('admin.receptions.show', $examination->id);
    } catch (\Exception $e) {
        DB::rollBack(); // Rollback in case of any error
        return redirect()->back()->with('error', __('messages.error') . ' ' . $e->getMessage());
    }
}


    public function edit($id)
    {
        // الحصول على السجل الذي نريد تعديله مع تفاصيله
        $examination = Examination::with('details')->findOrFail($id);

        // استرجاع البيانات الأخرى (العملاء، الفاحصين، الأنواع)
       $clients = Client::orderBy('created_at', 'desc')->get();

        $examiners = Examiner::all();
        $types = ['نظارات', 'عدسات'];
        $specials = ['عادي', 'مميز', 'مستعجل', 'تحضير', 'توصيل'];

        // تمرير البيانات إلى صفحة التعديل
        return view('admin.receptions.edit', compact('examination', 'clients', 'examiners', 'types','specials'));
    }



    public function show($id)
    {
        // الحصول على السجل الذي نريد تعديله مع تفاصيله
        $examination = Examination::with('details')->findOrFail($id);
      $specials = ['عادي', 'مميز', 'مستعجل', 'تحضير', 'توصيل'];
        // استرجاع البيانات الأخرى (العملاء، الفاحصين، الأنواع)
        $clients = Client::all();
        $examiners = Examiner::all();
        $types = ['نظارات', 'عدسات']; // أو يمكنك استرجاع الأنواع من قاعدة البيانات إذا كانت متغيرة

        // تمرير البيانات إلى صفحة العرض
        return view('admin.receptions.show', compact('examination', 'clients', 'examiners', 'types','specials'));
    }


    public function show_examination(int $id)
    {
        // Retrieve the examination record by its ID
        $examination = Examination::where('id', $id)->with('details')->first();
        $client = $examination->client;


        // Fetch the data based on room_code for the specific examination
        $user = Auth::user();
        $data = Data::where('room_code', $user->room_code)->get();
        $rooms = Room::all();

        // Prepare an array to hold formatted data
        $formatted_data = [];

        // Format the data as needed (similar to the create method)
        foreach ($data as $item) {
            // Parse the right eye (data_rt)
            preg_match('/R\s*-\s*-\s*-\s*(\d+\.\d+)\s*-\s*-\s*-\s*(\d+\.\d+)\s*-\s*(\d+)/', $item->data_rt, $matches_rt);
            $formatted_rt = [
                'type' => 'R',
                'sph' => $matches_rt[1] ?? 'N/A',  // SPH: -4.50
                'cyl' => $matches_rt[2] ?? 'N/A',  // CYL: -1.00
                'axis' => $matches_rt[3] ?? 'N/A', // Axis: 175
                'lpd' => $item->lpd_rt ?? 'N/A',   // Lens Power Difference for right eye
                'add' => $item->add_rt ?? 'N/A',   // Addition for right eye
            ];

            // Parse the left eye (data_lt)
            preg_match('/L\s*-\s*-\s*-\s*(\d+\.\d+)\s*-\s*-\s*-\s*(\d+\.\d+)\s*-\s*(\d+)/', $item->data_lt, $matches_lt);
            $formatted_lt = [
                'type' => 'L',
                'sph' => $matches_lt[1] ?? 'N/A',  // SPH: -4.25
                'cyl' => $matches_lt[2] ?? 'N/A',  // CYL: -1.75
                'axis' => $matches_lt[3] ?? 'N/A', // Axis: 170
                'lpd' => $item->lpd_lt ?? 'N/A',   // Lens Power Difference for left eye
                'add' => $item->add_lt ?? 'N/A',   // Addition for left eye
            ];

            // Collect formatted data for each item
            $formatted_data[] = [
                'id' => $item->id,
                'room_code' => $item->room_code,
                'right_eye' => $formatted_rt,
                'left_eye' => $formatted_lt,
            ];
        }

        // Fetch other required data for the view
        $clients = Client::all(); // Fetch all clients
        $types = ['نظارات', 'عدسات']; // Example types
        $examiners = Examiner::all();

        // Return the formatted data along with the rest of the information to the view
        return view('admin.receptions.show', compact('examination', 'client', 'clients', 'types', 'examiners', 'formatted_data', 'rooms'));
    }

    public function update(Request $request, $id)
    {
        // Validate the request data
        $request->validate([
            'code' => 'required' . $id,
            'notes' => 'nullable|string|max:255',
            'type' => 'required|string',
            'special' => 'required|string',
            'client_id' => 'required|exists:clients,id',
        ]);

        // Update the examination table
        $examination = Examination::findOrFail($id);
        $examination->update($request->only(['code', 'notes', 'type','special', 'client_id']));

        $examination->details()->updateOrCreate(
            ['type' => 'R', 'examination_id' => $id],
            [
                'sph' => $request->input('rt_sph'),
                'cyl' => $request->input('rt_cyl'),
                'axis' => $request->input('rt_axis'),
                'lpd' => $request->input('rt_lpd'),
                'add' => $request->input('rt_add')
            ]
        );

        // Update or create LT details
        $examination->details()->updateOrCreate(
            ['type' => 'L', 'examination_id' => $id],
            [
                'sph' => $request->input('lt_sph'),
                'cyl' => $request->input('lt_cyl'),
                'axis' => $request->input('lt_axis'),
                'lpd' => $request->input('lt_lpd'),
                'add' => $request->input('lt_add')
            ]
        );

        session()->flash('success', __('messages.update'));
        return redirect()->route('admin.receptions.index');
    }







    public function updateStatus(Request $request, $id)
    {
        // العثور على الفحص باستخدام الـ ID
        $examination = Examination::findOrFail($id);

        // تحديث الحالة فقط إذا تم إرسالها في الـ request
        if ($request->has('status')) {
            $examination->status = $request->input('status');
        }

        // إضافة أي تحديثات أخرى على الـ Examination إذا كان هناك المزيد من الحقول.
        // مثال:
        // if ($request->has('other_field')) {
        //    $examination->other_field = $request->input('other_field');
        // }

        // حفظ التغييرات
        $examination->save();

        // تحقق إذا كانت الحالة قد تغيرت فعلاً
        if ($examination->wasChanged()) {
            // إرسال رسالة WhatsApp عبر الـ Queue فقط إذا كانت الحالة قد تغيرت
            SendWhatsappMessage::dispatch($examination);

            return redirect()->route('admin.receptions.index')->with('success', 'تم تحديث الحالة وسيتم إرسال رسالة WhatsApp.');
        }

        // إذا لم تتغير الحالة
        return redirect()->route('admin.receptions.index')->with('info', 'الحالة لم تتغير.');
    }


    public function updateSpecial(Request $request, $id)
    {
        // العثور على الفحص باستخدام الـ ID
        $examination = Examination::findOrFail($id);
        // تحديث حقل `special` فقط إذا تم إرساله في الـ request
        if ($request->has('special')) {
            $examination->special = $request->input('special');
        }
        // حفظ التغييرات
        $examination->save();
        // إذا لم تتغير القيمة
        return redirect()->back()->with('success', 'تم تحديث نوع الفحص بنجاح');
    }



    public function print($id)
    {
        $examination = Examination::with(['client', 'examiner'])->findOrFail($id);
        return view('admin.receptions.print', compact('examination'));
    }

    public function printSpecial($id)
    {
        $examination = Examination::with(['client', 'examiner'])->findOrFail($id);
        $specials = ['عادي', 'مميز', 'مستعجل', 'تحضير', 'توصيل'];

        // تأكد من أن الفحص مميز
        if ($examination->special) {
            return view('admin.receptions.print_special', compact('examination','specials'));
        } else {
            return view('admin.receptions.print', compact('examination','specials'));
        }
    }
}
