<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ThemeSettings;
use App\Services\ThemeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;

class ThemeSettingsController extends Controller
{
    /**
     * @var ThemeService
     */
    protected $themeService;

    public function __construct(ThemeService $themeService)
    {
        $this->middleware('permission:settings', ['only' => ['index', 'update', 'presets', 'applyPreset']]);
        $this->themeService = $themeService;
    }

    public function index()
    {
        $themeSettings = ThemeSettings::first() ?? ThemeSettings::create();
        return view('admin.settings.theme', compact('themeSettings'));
    }

    public function update(Request $request)
    {
        $request->validate([
            'sidebar_bg_color' => 'required|string',
            'sidebar_text_color' => 'required|string',
            'header_bg_color' => 'required|string',
            'header_text_color' => 'required|string',
            'primary_color' => 'required|string',
            'secondary_color' => 'required|string',
            'font_family' => 'required|string',
            'base_font_size' => 'required|string',
            'heading_font_size' => 'required|string',
            'menu_font_size' => 'required|string',
            'text_color' => 'required|string',
            'link_color' => 'required|string',
            'is_dark_mode' => 'boolean',
        ]);

        $themeSettings = ThemeSettings::first();
        if (!$themeSettings) {
            $themeSettings = new ThemeSettings();
        }

        // Update settings
        $themeSettings->sidebar_bg_color = $request->sidebar_bg_color;
        $themeSettings->sidebar_text_color = $request->sidebar_text_color;
        $themeSettings->header_bg_color = $request->header_bg_color;
        $themeSettings->header_text_color = $request->header_text_color;
        $themeSettings->primary_color = $request->primary_color;
        $themeSettings->secondary_color = $request->secondary_color;
        $themeSettings->font_family = $request->font_family;
        $themeSettings->base_font_size = $request->base_font_size;
        $themeSettings->heading_font_size = $request->heading_font_size;
        $themeSettings->menu_font_size = $request->menu_font_size;
        $themeSettings->text_color = $request->text_color;
        $themeSettings->link_color = $request->link_color;
        $themeSettings->is_dark_mode = $request->has('is_dark_mode');
        $themeSettings->save();

        session()->flash('success', __('messages.update'));
        return back();
    }

    /**
     * Display the preset themes page
     */
    public function presets()
    {
        $colorSchemes = $this->themeService->getPresets();
        return view('admin.settings.theme_presets', compact('colorSchemes'));
    }

    /**
     * Apply a preset theme
     */
    public function applyPreset(Request $request)
    {
        $request->validate([
            'theme' => 'required|string',
        ]);

        $selectedScheme = $this->themeService->getPreset($request->theme);

        if (!$selectedScheme) {
            session()->flash('error', __('messages.theme_not_found'));
            return back();
        }

        $themeSettings = ThemeSettings::first();
        if (!$themeSettings) {
            $themeSettings = new ThemeSettings();
        }

        // Apply theme settings
        $themeSettings->sidebar_bg_color = $selectedScheme['sidebar_bg_color'];
        $themeSettings->sidebar_text_color = $selectedScheme['sidebar_text_color'];
        $themeSettings->header_bg_color = $selectedScheme['header_bg_color'];
        $themeSettings->header_text_color = $selectedScheme['header_text_color'];
        $themeSettings->primary_color = $selectedScheme['primary_color'];
        $themeSettings->secondary_color = $selectedScheme['secondary_color'];
        $themeSettings->text_color = $selectedScheme['text_color'];
        $themeSettings->link_color = $selectedScheme['link_color'];
        $themeSettings->font_family = $selectedScheme['font_family'];
        $themeSettings->base_font_size = $selectedScheme['base_font_size'];
        $themeSettings->heading_font_size = $selectedScheme['heading_font_size'];
        $themeSettings->menu_font_size = $selectedScheme['menu_font_size'];
        $themeSettings->is_dark_mode = $selectedScheme['is_dark_mode'];
        $themeSettings->save();

        session()->flash('success', __('messages.theme_applied'));
        return back();
    }
}
