<?php

namespace App\Http\Controllers\Admin;

use App\Models\Role;
use App\Models\Room;
use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:users-list', ['only' => ['index']]);
        $this->middleware('permission:users-create', ['only' => ['create', 'store']]);
        $this->middleware('permission:users-edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:users-delete', ['only' => ['destroy']]);
    }

    public function index()
    {
        $users = User::with('roles')->get();
        return view('admin.users.index', compact('users'));
    }

    public function create()
    {
        $roles = Role::all();
        $rooms = Room::all();
        return view('admin.users.create', compact('roles', 'rooms'));
    }

    public function store(Request $request)
    {

        // التحقق من صحة البيانات المدخلة
        $request->validate([
            'name'     => 'required',
            'email'    => 'required|email:filter|unique:users,email',
            'password' => 'required|confirmed|min:8',
            'role'     => 'required',
            'phone'    => 'required|unique:users,phone',
            'address'  => 'nullable',
            'room_code'  => 'required', // تحقق من أن الغرفة موجودة
        ]);
        $user = User::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'password' => Hash::make($request->password),
            'phone'    => $request->phone,
            'address'  => $request->address,
            'room_code'  => $request->room_code, // تعيين الغرفة
        ]);

        // إضافة الدور للمستخدم
        $user->addRole($request->role);

        // رسالة نجاح
        session()->flash('success', __('messages.create'));

        // إعادة التوجيه إلى قائمة المستخدمين
        return redirect()->route('admin.users.index');
    }


    public function edit(int $id)
    {
        $roles = Role::all();
        $user = User::findOrFail($id);
        $rooms = Room::all();
        return view('admin.users.edit', compact('roles', 'user', 'rooms'));
    }

    public function update(Request $request, int $id)
    {
        $user = User::findOrFail($id);


        $request->validate([
            'name'     => 'required',
            'email'    => 'required|email:filter|unique:users,email,' . $user->id,
            'role'     => 'required',
            'phone'    => 'required|unique:users,phone,' . $user->id,
            'address'  => 'nullable',
            'room_code'  => 'required',
        ]);

        $user->update([
            'name'     => $request->name,
            'email'    => $request->email,
            'phone'    => $request->phone,
            'address'  => $request->address,
            'room_code'  => $request->room_code,
        ]);

        $user->roles()->sync([]);
        $user->addRole($request->role);

        session()->flash('success', __('messages.update'));
        return redirect()->route('admin.users.index');
    }

    public function destroy(int $id)
    {
        $user = User::findOrFail($id);
        $user->roles()->sync([]);
        $user->delete();

        session()->flash('success', __('messages.delete'));
        return redirect()->route('admin.users.index');
    }
}
