<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Laratrust\Contracts\LaratrustUser;
use Laratrust\Traits\HasRolesAndPermissions;

class User extends Authenticatable implements LaratrustUser
{
    use HasFactory, Notifiable, HasApiTokens, HasRolesAndPermissions;

    // Specify the database table explicitly
    protected $table = 'users';

    // Mass assignable attributes
    protected $fillable = [
        'name',
        'email',
        'password',
        'phone',
        'address',
        'picture',
        'room_code'
    ];

    // Append custom attributes
    protected $appends = [
        'picture_url'
    ];

    // Hidden attributes for arrays
    protected $hidden = [
        'password',
        'remember_token',
    ];

    // Attribute casting
    protected $casts = [
        'password' => 'hashed',
        'email_verified_at' => 'datetime',
    ];

    /**
     * Accessor for picture URL.
     *
     * @return string
     */
    public function getPictureUrlAttribute(): string
    {
        if (!empty($this->picture)) {
            return asset('uploads/' . $this->picture);
        }

        // Default user image
        return asset('assets/images/user.png');
    }

    /**
     * Define the relationship with the Room model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function room()
    {
        return $this->belongsTo(Room::class, 'room_code', 'room_code');
    }

    /**
     * Check if the user is a Super Admin.
     *
     * @return bool
     */
    public function isSuperAdmin(): bool
    {
        return $this->hasRole('superadmin'); // Replace 'super_admin' with your Super Admin role name
    }
}
