<?php

namespace App\Services;

use App\Models\ThemeSettings;
use Illuminate\Support\Facades\Config;

class ThemeService
{
    protected $themeSettings;

    public function __construct()
    {
        $defaults = Config::get('theme.defaults', [
            'sidebar_bg_color' => '#1e1e2d',
            'header_bg_color' => '#ffffff',
            'primary_color' => '#009ef7',
            'secondary_color' => '#E4E6EF',
            'text_color' => '#3F4254',
            'link_color' => '#009ef7',
            'font_family' => 'Tajawal, Arial, sans-serif',
            'base_font_size' => '14px',
            'heading_font_size' => '1.25rem',
            'menu_font_size' => '1rem',
            'is_dark_mode' => false,
        ]);

        $this->themeSettings = ThemeSettings::first() ?? new ThemeSettings($defaults);
    }

    /**
     * Get all theme settings
     *
     * @return array
     */
    public function all()
    {
        return $this->themeSettings->toArray();
    }

    /**
     * Get a specific theme setting
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function get($key, $default = null)
    {
        return $this->themeSettings->{$key} ?? $default;
    }

    /**
     * Check if dark mode is enabled
     *
     * @return bool
     */
    public function isDarkMode()
    {
        return (bool) $this->themeSettings->is_dark_mode;
    }

    /**
     * Get available theme presets
     *
     * @return array
     */
    public function getPresets()
    {
        return Config::get('theme.presets', []);
    }

    /**
     * Get a specific theme preset
     *
     * @param string $key
     * @return array|null
     */
    public function getPreset($key)
    {
        return Config::get("theme.presets.{$key}");
    }

    /**
     * Get CSS variables for the theme
     *
     * @return string
     */
    public function cssVariables()
    {
        return "
            :root {
                --sidebar-bg-color: {$this->themeSettings->sidebar_bg_color};
                --header-bg-color: {$this->themeSettings->header_bg_color};
                --primary-color: {$this->themeSettings->primary_color};
                --secondary-color: {$this->themeSettings->secondary_color};
                --text-color: {$this->themeSettings->text_color};
                --link-color: {$this->themeSettings->link_color};
                --font-family: {$this->themeSettings->font_family};
                --base-font-size: {$this->themeSettings->base_font_size};
                --heading-font-size: {$this->themeSettings->heading_font_size};
                --menu-font-size: {$this->themeSettings->menu_font_size};
            }
        ";
    }

    /**
     * Get inline styles for using in style attributes
     *
     * @param string $element Element type (sidebar, header, text, etc)
     * @return string
     */
    public function inlineStyle($element)
    {
        $styles = [];

        switch ($element) {
            case 'sidebar':
                $styles = [
                    'background-color' => $this->themeSettings->sidebar_bg_color,
                    'color' => '#ffffff',
                ];
                break;
            case 'header':
                $styles = [
                    'background-color' => $this->themeSettings->header_bg_color,
                    'color' => $this->themeSettings->text_color,
                ];
                break;
            case 'text':
                $styles = [
                    'color' => $this->themeSettings->text_color,
                    'font-family' => $this->themeSettings->font_family,
                    'font-size' => $this->themeSettings->base_font_size,
                ];
                break;
            case 'heading':
                $styles = [
                    'color' => $this->themeSettings->text_color,
                    'font-family' => $this->themeSettings->font_family,
                    'font-size' => $this->themeSettings->heading_font_size,
                ];
                break;
            case 'link':
                $styles = [
                    'color' => $this->themeSettings->link_color,
                ];
                break;
            case 'primary-button':
                $styles = [
                    'background-color' => $this->themeSettings->primary_color,
                    'color' => '#ffffff',
                    'border' => 'none',
                ];
                break;
            case 'secondary-button':
                $styles = [
                    'background-color' => $this->themeSettings->secondary_color,
                    'color' => '#ffffff',
                    'border' => 'none',
                ];
                break;
        }

        return $this->stylesToString($styles);
    }

    /**
     * Convert style array to inline style string
     *
     * @param array $styles
     * @return string
     */
    protected function stylesToString($styles)
    {
        $styleString = '';

        foreach ($styles as $property => $value) {
            $styleString .= "{$property}: {$value}; ";
        }

        return rtrim($styleString);
    }
}
