<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Faker\Factory as Faker;

class ExaminationSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        // Increase memory limit (if necessary)
        ini_set('memory_limit', '512M'); // Adjust as needed

        // Disable query log for better performance
        DB::disableQueryLog();

        // Start transaction for better performance and consistency
        DB::beginTransaction();

        try {
            $faker = Faker::create(); // Create a Faker instance

            $batchSize = 500; // Reduced batch size for better memory management
            $totalRecords = 1000000; // Number of records to insert
            $examinations = []; // Temporary array to store batch data

            for ($i = 1; $i <= $totalRecords; $i++) {
                // Prepare each examination record
                $examinations[] = [
                    'code' => 'EXM' . str_pad($i, 5, '0', STR_PAD_LEFT), // EXM00001, EXM00002, etc.
                    'notes' => $faker->sentence, // Fake sentence for notes
                    'type' => $faker->randomElement(['نظارات', 'عدسات']), // Randomly choose between 'نظارات' and 'عدسات'
                    'make_from' => $faker->randomElement(['examinations', 'receptions']), // Randomly choose between 'examinations' and 'receptions'
                    'status' => $this->getRandomStatus(), // Random status function
                    'special' => false,
                    'category' => $this->getRandomCategory(), // Random category function
                    'room_code' => 'Room' . rand(1, 100), // Random room code
                    'client_id' => rand(1, 1000), // Random client ID
                    'examiner_id' => rand(1, 500), // Random examiner ID
                    'create_by' => rand(1, 50), // Random user ID
                    'update_by' => rand(1, 50), // Random user ID
                    'created_at' => $faker->dateTimeThisDecade->format('Y-m-d H:i:s'), // Fake random date within the last 10 years
                    'updated_at' => $faker->dateTimeThisDecade->format('Y-m-d H:i:s'), // Fake random date within the last 10 years
                ];

                // Insert data in batches
                if ($i % $batchSize == 0 || $i == $totalRecords) {
                    DB::table('examinations')->insert($examinations);
                    $examinations = []; // Clear the array for the next batch
                }
            }

            // Commit the transaction after successful insert
            DB::commit();
            Log::info('Examination seeder completed successfully.');
        } catch (\Exception $e) {
            // Rollback the transaction if there is an error
            DB::rollBack();
            Log::error('Seeding failed: ' . $e->getMessage());
        }
    }

    /**
     * Get a random status for the examination.
     *
     * @return string
     */
    private function getRandomStatus()
    {
        $statuses = ['في انتظار التاكيد', 'ارسال الي المشغل', 'تم التجهيز', 'تم الاستلام', 'الغاء'];
        return $statuses[array_rand($statuses)];
    }

    /**
     * Get a random category for the examination.
     *
     * @return string|null
     */
    private function getRandomCategory()
    {
        $categories = ['a', 'b', 'c', 'd'];
        return $categories[array_rand($categories)]; // You can modify this logic if you want null values randomly
    }
}

